<?php

declare(strict_types=1);

namespace Drupal\Tests\file\Kernel;

use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\Core\Field\FieldItemInterface;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Field\FieldStorageDefinitionInterface;
use Drupal\entity_test\Entity\EntityTest;
use Drupal\field\Entity\FieldConfig;
use Drupal\field\Entity\FieldStorageConfig;
use Drupal\file\Entity\File;
use Drupal\file\Plugin\Field\FieldType\FileItem;
use Drupal\Tests\field\Kernel\FieldKernelTestBase;
use Drupal\user\Entity\Role;
use PHPUnit\Framework\Attributes\Group;
use PHPUnit\Framework\Attributes\RunTestsInSeparateProcesses;

/**
 * Tests using entity fields of the file field type.
 */
#[Group('file')]
#[RunTestsInSeparateProcesses]
class FileItemTest extends FieldKernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = ['file'];

  /**
   * Created file entity.
   *
   * @var \Drupal\file\Entity\File
   */
  protected $file;

  /**
   * Directory where the sample files are stored.
   *
   * @var string
   */
  protected $directory;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->installEntitySchema('user');
    $this->installConfig(['user']);
    // Give anonymous users permission to access content, so they can view and
    // download public files.
    $anonymous_role = Role::load(Role::ANONYMOUS_ID);
    $anonymous_role->grantPermission('access content');
    $anonymous_role->save();

    $this->installEntitySchema('file');
    $this->installSchema('file', ['file_usage']);

    FieldStorageConfig::create([
      'field_name' => 'file_test',
      'entity_type' => 'entity_test',
      'type' => 'file',
      'cardinality' => FieldStorageDefinitionInterface::CARDINALITY_UNLIMITED,
    ])->save();
    $this->directory = $this->getRandomGenerator()->name(8);
    FieldConfig::create([
      'entity_type' => 'entity_test',
      'field_name' => 'file_test',
      'bundle' => 'entity_test',
      'settings' => ['file_directory' => $this->directory],
    ])->save();
    file_put_contents('public://example.txt', $this->randomMachineName());
    $this->file = File::create([
      'uri' => 'public://example.txt',
    ]);
    $this->file->save();
  }

  /**
   * Tests using entity fields of the file field type.
   */
  public function testFileItem(): void {
    // Check that the selection handler was automatically assigned to
    // 'default:file'.
    $field_definition = FieldConfig::load('entity_test.entity_test.file_test');
    $handler_id = $field_definition->getSetting('handler');
    $this->assertEquals('default:file', $handler_id);

    // Create a test entity with the test file field.
    $entity = EntityTest::create();
    $entity->file_test->target_id = $this->file->id();
    $entity->file_test->display = 1;
    $entity->file_test->description = $description = $this->randomMachineName();
    $entity->name->value = $this->randomMachineName();
    $entity->save();

    $entity = EntityTest::load($entity->id());
    $this->assertInstanceOf(FieldItemListInterface::class, $entity->file_test);
    $this->assertInstanceOf(FieldItemInterface::class, $entity->file_test[0]);
    $this->assertEquals($this->file->id(), $entity->file_test->target_id);
    $this->assertEquals(1, $entity->file_test->display);
    $this->assertEquals($description, $entity->file_test->description);
    $this->assertEquals($this->file->getFileUri(), $entity->file_test->entity->getFileUri());
    $this->assertEquals($this->file->id(), $entity->file_test->entity->id());
    $this->assertEquals($this->file->uuid(), $entity->file_test->entity->uuid());

    // Make sure the computed files reflects updates to the file.
    file_put_contents('public://example-2.txt', $this->randomMachineName());
    $file2 = File::create([
      'uri' => 'public://example-2.txt',
    ]);
    $file2->save();

    $entity->file_test->target_id = $file2->id();
    $this->assertEquals($entity->file_test->entity->id(), $file2->id());
    $this->assertEquals($entity->file_test->entity->getFileUri(), $file2->getFileUri());

    // Test the deletion of an entity having an entity reference field targeting
    // a non-existing entity.
    $file2->delete();
    $entity->delete();

    // Test the generateSampleValue() method.
    $entity = EntityTest::create();
    $entity->file_test->generateSampleItems();
    $this->entityValidateAndSave($entity);
    // Verify that the sample file was stored in the correct directory.
    $uri = $entity->file_test->entity->getFileUri();

    /** @var \Drupal\Core\StreamWrapper\StreamWrapperManagerInterface $stream_wrapper_manager */
    $stream_wrapper_manager = \Drupal::service('stream_wrapper_manager');

    $this->assertEquals($this->directory, dirname($stream_wrapper_manager::getTarget($uri)));

    // Make sure the computed files reflects updates to the file.
    file_put_contents('public://example-3.txt', $this->randomMachineName());
    // Test unsaved file entity.
    $file3 = File::create([
      'uri' => 'public://example-3.txt',
    ]);
    $display = \Drupal::service('entity_display.repository')
      ->getViewDisplay('entity_test', 'entity_test');
    $display->setComponent('file_test', [
      'label' => 'above',
      'type' => 'file_default',
      'weight' => 1,
    ])->save();
    $entity = EntityTest::create();
    $entity->file_test = ['entity' => $file3];
    $uri = $file3->getFileUri();
    $output = \Drupal::entityTypeManager()
      ->getViewBuilder('entity_test')
      ->view($entity, 'default');
    \Drupal::service('renderer')->renderRoot($output);
    $this->assertTrue(!empty($entity->file_test->entity));
    $this->assertEquals($uri, $entity->file_test->entity->getFileUri());

    // Test file URIs with empty and custom directories.
    $this->validateFileUriForDirectory(
      '', 'public://'
    );
    $this->validateFileUriForDirectory(
      'custom_directory/subdir', 'public://custom_directory/subdir/'
    );
  }

  /**
   * Tests file URIs generated for a given file directory.
   *
   * @param string $file_directory
   *   The file directory to test (e.g., empty or 'custom_directory/subdir').
   * @param string $expected_start
   *   The expected starting string of the file URI (e.g., 'public://').
   */
  private function validateFileUriForDirectory(string $file_directory, string $expected_start): void {
    // Mock the field definition with the specified file directory.
    $definition = $this->createMock(FieldDefinitionInterface::class);
    $definition->expects($this->any())
      ->method('getSettings')
      ->willReturn([
        'file_extensions' => 'txt',
        'file_directory' => $file_directory,
        'uri_scheme' => 'public',
        'display_default' => TRUE,
      ]);

    // Generate a sample file value.
    $value = FileItem::generateSampleValue($definition);
    $this->assertNotEmpty($value);

    // Load the file entity and get its URI.
    $fid = $value['target_id'];
    $file = File::load($fid);
    $fileUri = $file->getFileUri();

    // Verify the file URI starts with the expected protocol and structure.
    $this->assertStringStartsWith($expected_start, $fileUri);
    $this->assertMatchesRegularExpression('#^' . preg_quote($expected_start, '#') . '[^/]+#', $fileUri);
  }

  /**
   * Tests sample value generation with actual allowed file extensions.
   */
  public function testGenerateSampleValue(): void {
    /** @var \Drupal\field\Entity\FieldConfig $field_definition */
    $field_definition = FieldConfig::loadByName('entity_test', 'entity_test', 'file_test');
    $field_definition->setSetting('file_extensions', 'pdf');
    $field_definition->save();

    $class = $field_definition->getItemDefinition()->getClass();
    $value = call_user_func("$class::generateSampleValue", $field_definition);
    /** @var \Drupal\file\FileInterface $file */
    $file = File::load($value['target_id']);
    $this->assertStringEndsWith('.pdf', $file->getFileUri());
  }

}
