<?php

declare(strict_types=1);

namespace Drupal\Tests\node\Kernel\Views;

use Drupal\node\Entity\Node;
use Drupal\Tests\user\Traits\UserCreationTrait;
use Drupal\Tests\views\Kernel\ViewsKernelTestBase;
use Drupal\views\Tests\ViewTestData;
use Drupal\views\Views;
use PHPUnit\Framework\Attributes\Group;
use PHPUnit\Framework\Attributes\RunTestsInSeparateProcesses;

/**
 * Tests the argument_node_uid_revision handler.
 */
#[Group('node')]
#[RunTestsInSeparateProcesses]
class ArgumentUidRevisionTest extends ViewsKernelTestBase {
  use UserCreationTrait;

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'node',
    'field',
    'node_test_views',
  ];

  /**
   * {@inheritdoc}
   */
  public static $testViews = ['test_argument_node_uid_revision'];

  /**
   * {@inheritdoc}
   */
  protected function setUp($import_test_views = TRUE): void {
    parent::setUp($import_test_views);

    $this->installEntitySchema('node');
    $this->installSchema('node', ['node_access']);
    $this->installEntitySchema('user');
    $this->installConfig(['node', 'field']);

    ViewTestData::createTestViews(static::class, ['node_test_views']);
  }

  /**
   * Tests the node_uid_revision argument.
   */
  public function testArgument(): void {
    $expected_result = [];

    $author = $this->createUser();
    $no_author = $this->createUser();

    // Create one node, with the author as the node author.
    $node1 = Node::create([
      'type' => 'default',
      'title' => $this->randomMachineName(),
    ]);
    $node1->setOwner($author);
    $node1->save();
    $expected_result[] = ['nid' => $node1->id()];

    // Create one node of which an additional revision author will be the
    // author.
    $node2 = Node::create([
      'type' => 'default',
      'title' => $this->randomMachineName(),
    ]);
    $node2->setRevisionUserId($no_author->id());
    $node2->save();
    $expected_result[] = ['nid' => $node2->id()];

    // Force to add a new revision.
    $node2->setNewRevision(TRUE);
    $node2->setRevisionUserId($author->id());
    $node2->save();

    // Create one  node on which the author has neither authorship of revisions
    // or the main node.
    $node3 = Node::create([
      'type' => 'default',
      'title' => $this->randomMachineName(),
    ]);
    $node3->setOwner($no_author);
    $node3->save();

    $view = Views::getView('test_argument_node_uid_revision');
    $view->initHandlers();
    $view->setArguments(['uid_revision' => $author->id()]);

    $this->executeView($view);
    $this->assertIdenticalResultset($view, $expected_result, ['nid' => 'nid']);
  }

}
